<?php
/*
Widget Name: Lightbox Builder
Description: Create a custom lightbox with Page Builder content and styling.
Author: SiteOrigin
Author URI: https://siteorigin.com
Documentation: https://siteorigin.com/premium-documentation/plugin-addons/lightbox-builder/
*/

if (
	! class_exists( 'SiteOrigin_Widget' ) ||
	! class_exists( 'SiteOrigin_Panels' )
) {
	return;
}

class SiteOrigin_Premium_Widget_Lightbox_Builder extends SiteOrigin_Widget {
	private $button_class;
	public function __construct() {
		parent::__construct(
			'so-premium-lightbox-builder',
			__( 'SiteOrigin Lightbox Builder', 'siteorigin-premium' ),
			array(
				'description' => __( 'Create a custom lightbox with Page Builder content and styling.', 'siteorigin-premium' ),
				'help' => 'https://siteorigin.com/premium-documentation/plugin-addons/lightbox-builder/',
				'panels',
				'panels_title' => false,
				'has_preview' => false,
			),
			array(),
			false,
			plugin_dir_path( __FILE__ )
		);

		add_filter( 'siteorigin_widgets_block_exclude_widget', array( $this, 'exclude_from_widgets_block_cache' ), 10, 2 );
	}

	public function exclude_from_widgets_block_cache( $exclude, $widget_class ) {
		if ( $widget_class === 'SiteOrigin_Premium_Widget_Lightbox_Builder' ) {
			$exclude = true;
		}

		return $exclude;
	}

	public function get_widget_form() {
		return array(
			'title' => array(
				'type' => 'text',
				'label' => __( 'Title', 'siteorigin-premium' ),
				'description' => __( 'The title is output inside of the lightbox, and helps with accessability.', 'siteorigin-premium' ),
			),
			'content' => array(
				'type' => 'builder',
				'label' => __( 'Content', 'siteorigin-premium' ),
			),
			'button_open' => array(
				'type' => 'widget',
				'label' => __( 'Open Button', 'siteorigin-premium' ),
				'hide' => true,
				'class' => 'SiteOrigin_Widget_Button_Widget',
				'form_filter' => array( $this, 'filter_button_widget' ),
			),
			'close_icon' => array(
				'type' => 'section',
				'label' => __( 'Close Icon', 'siteorigin-premium' ),
				'hide' => true,
				'fields' => array(
					'icon' => array(
						'type' => 'icon',
						'label' => __( 'Icon', 'siteorigin-premium' ),
						'default' => 'fontawesome-sow-fas-xmark',
					),
					'color' => array(
						'type' => 'color',
						'label' => __( 'Color', 'siteorigin-premium' ),
						'default' => '#2d2d2d',
					),
					'size' => array(
						'type' => 'measurement',
						'label' => __( 'Size', 'siteorigin-premium' ),
						'default' => '20px',
					),
					'position' => array(
						'type' => 'select',
						'label' => __( 'Position', 'siteorigin-premium' ),
						'default' => 'top-right',
						'options' => array(
							'top-right' => __( 'Top Right', 'siteorigin-premium' ),
							'top-left' => __( 'Top Left', 'siteorigin-premium' ),
						),
					),
				),
			),
			'design' => array(
				'type' => 'section',
				'label' => __( 'Design', 'siteorigin-premium' ),
				'hide' => true,
				'fields' => array(
					'title' => array(
						'type' => 'section',
						'label' => __( 'Title', 'siteorigin-premium' ),
						'hide' => true,
						'fields' => array(
							'tag' => array(
								'type' => 'select',
								'label' => __( 'Tag', 'siteorigin-premium' ),
								'default' => 'h4',
								'options' => array(
									'h1' => 'H1',
									'h2' => 'H2',
									'h3' => 'H3',
									'h4' => 'H4',
									'h5' => 'H5',
									'h6' => 'H6',
									'p' => 'P',
								),
							),
							'font' => array(
								'type' => 'font',
								'label' => __( 'Font', 'siteorigin-premium' ),
							),
							'color' => array(
								'type' => 'color',
								'label' => __( 'Color', 'siteorigin-premium' ),
								'default' => '#000',
							),
							'size' => array(
								'type' => 'measurement',
								'label' => __( 'Size', 'siteorigin-premium' ),
							),
							'margin_after' => array(
								'type' => 'measurement',
								'label' => __( 'Margin After Title', 'siteorigin-premium' ),
							),
						),
					),
					'content' => array(
						'type' => 'section',
						'label' => __( 'Content', 'siteorigin-premium' ),
						'hide' => true,
						'fields' => array(
							'background_color' => array(
								'type' => 'color',
								'label' => __( 'Background Color', 'siteorigin-premium' ),
								'default' => '#ffffff',
								'alpha' => true,
							),
							'border_width' => array(
								'type' => 'measurement',
								'label' => __( 'Border Width', 'siteorigin-premium' ),
							),
							'border_color' => array(
								'type' => 'color',
								'label' => __( 'Border Color', 'siteorigin-premium' ),
								'default' => '#000000',
								'alpha' => true,
							),
							'padding' => array(
								'type' => 'multi-measurement',
								'label' => __( 'Padding', 'siteorigin-premium' ),
								'default' => '30px 30px 30px 30px',
								'autofill' => true,
								'measurements' => array(
									'top' => __( 'Left', 'siteorigin-premium' ),
									'right' => __( 'Right', 'siteorigin-premium' ),
									'bottom' => __( 'Bottom', 'siteorigin-premium' ),
									'left' => __( 'Left', 'siteorigin-premium' ),
								),
							),
						),
					),
					'lightbox' => array(
						'type' => 'section',
						'label' => __( 'Lightbox', 'siteorigin-premium' ),
						'hide' => true,
						'fields' => array(
							'background_color' => array(
								'type' => 'color',
								'label' => __( 'Background Color', 'siteorigin-premium' ),
								'default' => 'rgba(0,0,0,0.8)',
								'alpha' => true,
							),
							'height' => array(
								'type' => 'measurement',
								'label' => __( 'Height', 'siteorigin-premium' ),
								'default' => '70vh',
							),
							'max_width' => array(
								'type' => 'measurement',
								'label' => __( 'Maximum Width', 'siteorigin-premium' ),
								'default' => '60vw',
							),
						),
					),
				),
			),
		);
	}

	public function filter_button_widget( $form_fields ) {
		$form_fields['text']['default'] = __( 'Open Lightbox', 'siteorigin-premium' );
		unset( $form_fields['url'] );
		unset( $form_fields['new_window'] );
		unset( $form_fields['download'] );

		// Change defaults.
		$form_fields['design']['fields']['button_color']['default'] = '#239cff';
		$form_fields['design']['fields']['hover_background_color']['alpha'] = true;
		$form_fields['design']['fields']['hover_background_color']['default'] = 'rgba(35, 156, 255, 0.8)';

		return $form_fields;
	}


	public function get_less_variables( $instance ) {
		if ( empty( $instance ) ) {
			return array();
		}

		$content_padding = ! empty( $instance['design']['content']['padding'] ) ? $instance['design']['content']['padding'] : '';
		$inverted_right_padding = '';
		$inverted_left_padding = '';

		// To prevent display issues with full width stretch rows,
		// we need to invert the left and right padding.
		if ( ! empty( $content_padding ) ) {
			$content_padding_parts = explode( ' ', $content_padding );
			$content_padding_left = $content_padding_parts[3];
			$content_padding_right = $content_padding_parts[1];

			$inverted_left_padding = '-' . $content_padding_left;
			$inverted_right_padding = '-' . $content_padding_right;
		}

		$vars = array(
			'title_color' => ! empty( $instance['design']['title']['color'] ) ? $instance['design']['title']['color'] : '',
			'title_size' => ! empty( $instance['design']['title']['size'] ) ? $instance['design']['title']['size'] : '',
			'title_margin_after' => ! empty( $instance['design']['title']['margin_after'] ) ? $instance['design']['title']['margin_after'] : '',
			'content_padding' => $content_padding,
			'content_padding_left_inverted' => $inverted_left_padding,
			'content_padding_right_inverted' => $inverted_right_padding,
			'content_background_color' => ! empty( $instance['design']['content']['background_color'] ) ? $instance['design']['content']['background_color'] : '',
			'lightbox_height' => ! empty( $instance['design']['lightbox']['height'] ) ? $instance['design']['lightbox']['height'] : '',
			'lightbox_background_color' => ! empty( $instance['design']['lightbox']['background_color'] ) ? $instance['design']['lightbox']['background_color'] : '',
			'lightbox_max_width' => ! empty( $instance['design']['lightbox']['max_width'] ) ? $instance['design']['lightbox']['max_width'] : '',
			'close_icon_color' => ( ! empty( $instance['close_icon'] ) && is_array( $instance['close_icon'] ) && ! empty( $instance['close_icon']['color'] ) ) ? $instance['close_icon']['color'] : '#2d2d2d',
			'close_icon_size' => ( ! empty( $instance['close_icon'] ) && is_array( $instance['close_icon'] ) && ! empty( $instance['close_icon']['size'] ) ) ? $instance['close_icon']['size'] : '20px',
		);

		$border = ! empty( $instance['design']['content']['border_width'] ) ? $instance['design']['content']['border_width'] : '1px';
		$border .= ' solid ';
		$vars['content_border'] = ! empty( $instance['design']['content']['border_color'] ) ? $instance['design']['content']['border_color'] : 'transparent';

		if ( ! empty( $instance['design']['title']['font'] ) ) {
			if ( ! empty( $text['font'] ) ) {
				$font = siteorigin_widget_get_font( $text['font'] );
				$vars['title_font'] = $font['family'];

				if ( ! empty( $font['weight'] ) ) {
					$vars['title_style'] = $font['style'];
					$vars['title_weight'] = $font['weight_raw'];
				}
			}
		}

		return $vars;
	}

	/**
	 * Renders the "Open" button for the lightbox dialog.
	 *
	 * This method generates the HTML for the button that opens the lightbox dialog.
	 * It uses the SiteOrigin Button Widget to render the button with the provided settings.
	 * Filters are temporarily added to prevent block editor issues, and to add custom
	 * classes.
	 *
	 * @param string $dialog_id The unique ID of the lightbox dialog.
	 * @param array  $settings The settings for the button widget, including its design
	 * and behavior.
	 *
	 * @return string The rendered HTML for the "Open" button, or an empty string if
	 * the button widget is not available.
	 */
	private function render_open_button( $dialog_id, $settings ) {
		global $wp_widget_factory;

		// Don't attempt to render button if the widget isn't available.
		if ( ! isset( $wp_widget_factory->widgets['SiteOrigin_Widget_Button_Widget'] ) ) {
			return '';
		}

		$button_widget = $wp_widget_factory->widgets['SiteOrigin_Widget_Button_Widget'];

		add_filter( 'siteorigin_widgets_button_attributes', array( $this, 'add_button_classes' ) );

		add_filter( 'siteorigin_widgets_is_preview', '__return_false' );

		ob_start();
		$this->button_class = 'open';
		$settings['url'] = "#$dialog_id";
		$button_widget->widget( array(), $settings );
		$button_open = ob_get_clean();

		remove_filter( 'siteorigin_widgets_is_preview', '__return_false' );
		remove_filter( 'siteorigin_widgets_button_attributes', array( $this, 'add_button_classes' ) );

		return $button_open;
	}

	public function get_template_variables( $instance, $args ) {
		if ( empty( $instance ) ) {
			return array();
		}

		if ( defined( 'SITEORIGIN_PANELS_VERSION' ) ) {
			$content = siteorigin_panels_render(
				'wSoPremiumLightboxBuilder-' . sanitize_text_field( $args['widget_id'] ),
				true,
				$instance['content']
			);
		}

		$dialog_id = uniqid( 'siteorigin-lightbox-dialog-' );

		$button_open = $this->render_open_button( $dialog_id, $instance['button_open'] );

		// Generate close icon HTML.
		$close_icon_html = '';
		$close_icon_value = 'fontawesome-sow-fas-xmark';
		if (
			is_array( $instance['close_icon'] ) &&
			! empty( $instance['close_icon']['icon'] )
		) {
			$close_icon_value = $instance['close_icon']['icon'];
		}

		if ( ! empty( $close_icon_value ) ) {
			$close_icon_html = siteorigin_widget_get_icon( $close_icon_value );

			wp_enqueue_style( 'siteorigin-accessibility' );
		}

		return array(
			'content' => empty( $content ) ? '' : $content,
			'title' => empty( $instance['title'] ) ? '' : $instance['title'],
			'title_tag' => SiteOrigin_Premium_Utility::single()->validate_tag(
				$instance['design']['title']['tag'],
				'h4'
			),
			'button_open' => $button_open,
			'close_icon' => $close_icon_html,
			'close_icon_position' => ( ! empty( $instance['close_icon'] ) && is_array( $instance['close_icon'] ) && ! empty( $instance['close_icon']['position'] ) ) ? $instance['close_icon']['position'] : 'top-right',
			'dialog_id' => $dialog_id,
		);
	}

	public function add_button_classes( $attributes ) {
		$attributes['class'] .= " siteorigin-lightbox-btn-$this->button_class";

		return $attributes;
	}
}

siteorigin_widget_register( 'so-premium-lightbox-builder', __FILE__, 'SiteOrigin_Premium_Widget_Lightbox_Builder' );
