<?php
class SiteOrigin_Premium_Plugin_Toggle_Visibility_Page_Builder {
	private $addon;
	private $migration;

	public function __construct() {
		$this->addon = SiteOrigin_Premium_Plugin_Toggle_Visibility::single();
		$this->migration = SiteOrigin_Premium_Plugin_Toggle_Visibility_Migrations::single();

		add_filter( 'siteorigin_panels_row_style_groups', array( $this, 'style_group' ), 10, 3 );
		add_filter( 'siteorigin_panels_row_style_fields', array( $this, 'style_fields' ), 10, 3 );
		add_filter( 'siteorigin_panels_widget_style_groups', array( $this, 'style_group' ), 10, 3 );
		add_filter( 'siteorigin_panels_widget_style_fields', array( $this, 'style_fields' ), 10, 3 );
		add_filter( 'siteorigin_panels_css_object', array( $this, 'add_row_widget_visibility_css' ), 10, 4 );

		if ( $this->migration->pb_uses_new_widget_row_hiding ) {
			add_filter( 'siteorigin_panels_output_row', array( $this, 'maybe_hide_row' ), 10, 2 );
			add_filter( 'siteorigin_panels_output_widget', array( $this, 'maybe_hide_widget' ), 10, 2 );
		}
	}

	public static function single() {
		static $single;

		return empty( $single ) ? $single = new self() : $single;
	}

	public function style_group( $groups, $post_id, $args ) {
		$groups['toggle'] = array(
			'name' => __( 'Toggle Visibility', 'siteorigin-premium' ),
			'priority' => 30,
		);

		return $groups;
	}

	public function style_fields( $fields, $post_id, $args ) {
		if ( current_filter() == 'siteorigin_panels_row_style_fields' ) {
			$fields['disable_row'] = array(
				// Adding empty 'name' field to avoid 'Undefined index' notices in PB due to always expecting
				// name 'field' in siteorigin-panels\inc\styles-admin.php:L145.
				'name' => '',
				'label' => __( 'Hide Row on All Devices', 'siteorigin-premium' ),
				'type' => 'checkbox',
				'group' => 'toggle',
				'priority' => 10,
			);
		} else {
			$fields['disable_widget'] = array(
				'name' => '',
				'label' => __( 'Hide Widget on All Devices', 'siteorigin-premium' ),
				'type' => 'checkbox',
				'group' => 'toggle',
				'priority' => 10,
			);
		}

		$fields['disable_desktop'] = array(
			'name' => '',
			'label' => __( 'Hide on Desktop', 'siteorigin-premium' ),
			'type' => 'checkbox',
			'group' => 'toggle',
			'priority' => 20,
		);

		$fields['disable_tablet'] = array(
			'name' => '',
			'label' => __( 'Hide on Tablet', 'siteorigin-premium' ),
			'type' => 'checkbox',
			'group' => 'toggle',
			'priority' => 30,
		);

		$fields['disable_mobile'] = array(
			'name' => '',
			'label' => __( 'Hide on Mobile', 'siteorigin-premium' ),
			'type' => 'checkbox',
			'group' => 'toggle',
			'priority' => 40,
		);

		$fields['disable_logged_out'] = array(
			'name' => '',
			'label' => __( 'Hide When Logged Out', 'siteorigin-premium' ),
			'type' => 'checkbox',
			'group' => 'toggle',
			'priority' => 50,
		);

		$fields['disable_logged_in'] = array(
			'name' => '',
			'label' => __( 'Hide When Logged In', 'siteorigin-premium' ),
			'type' => 'checkbox',
			'group' => 'toggle',
			'priority' => 60,
		);

		$fields['toggle_scheduling'] = array(
			'name' => __( 'Scheduling', 'siteorigin-premium' ),
			'type' => 'checkbox',
			'group' => 'toggle',
			'priority' => 70,
		);
		$toggle_fields = array();

		$toggle_fields['toggle_display'] = array(
			'name' => __( 'Display', 'siteorigin-premium' ),
			'type' => 'radio',
			'group' => 'toggle',
			'default' => 'show',
			'priority' => 80,
			'options' => array(
				'show' => 'Show',
				'hide' => 'Hide',
			),
		);

		$toggle_fields['toggle_date_from'] = array(
			'name' => __( 'Date From', 'siteorigin-premium' ),
			'type' => 'text',
			'group' => 'toggle',
			'priority' => 90,
		);

		$toggle_fields['toggle_date_to'] = array(
			'name' => __( 'Date To', 'siteorigin-premium' ),
			'type' => 'text',
			'group' => 'toggle',
			'priority' => 100,
		);

		// Is a version of Page Builder that supports the Toggle field active?
		if ( ! empty( $this->migration->pb_uses_toggle ) ) {
			$fields['toggle_scheduling']['type'] = 'toggle';
			$fields['toggle_scheduling']['fields'] = $toggle_fields;
		} else {
			$fields['toggle_scheduling']['label'] = __( 'Enable', 'siteorigin-premium' );
			$fields = array_merge(
				$fields,
				$toggle_fields
			);
		}

		// Is a pre-multi-select field version of PB active?
		// If so, don't add User Role setting.
		if ( ! $this->migration->pb_user_roles_support ) {
			return $fields;
		}

		$fields['toggle_user_role'] = array(
			'name' => __( 'User Roles', 'siteorigin-premium' ),
			'type' => 'toggle',
			'group' => 'toggle',
			'priority' => 71,
			'fields' => array(
				'toggle_display' => array(
					'name' => __( 'Display', 'siteorigin-premium' ),
					'type' => 'radio',
					'group' => 'toggle',
					'default' => 'show',
					'priority' => 80,
					'options' => array(
						'show' => 'Show',
						'hide' => 'Hide',
					),
				),
				'roles' => array(
					'name' => __( 'Roles', 'siteorigin-premium' ),
					'type' => 'multi-select',
					'options' => $this->addon->get_roles(),
				)
			),
		);

		$this->addon->prepare_exempt_roles();
		if ( ! empty( $this->addon->exempt_roles ) ) {
			$fields['toggle_user_role']['fields']['roles']['description'] = SiteOrigin_Premium_Utility::get_user_roles_description(
				$this->addon->exempt_roles
			);
		}

		return $fields;
	}

	/**
	 * Add row/widget CSS for device specific visibility.
	 */
	public function add_row_widget_visibility_css( $css, $panels_data, $post_id, $layout_data ) {
		$panels_tablet_width = siteorigin_panels_setting( 'tablet-width' );
		$panels_mobile_width = siteorigin_panels_setting( 'mobile-width' );
		$desktop_breakpoint = ( $panels_tablet_width === '' ? $panels_mobile_width : $panels_tablet_width ) + 1;
		$tablet_min_width = $panels_mobile_width + 1;

		foreach ( $layout_data as $ri => $row ) {
			// Check if row is disabled on desktop.
			if ( ! empty( $row['style']['disable_desktop'] ) ) {
				$css->add_row_css( $post_id, $ri, null, array(
					'display' => 'none',
				), ":$desktop_breakpoint" );
			}

			// Check if row is disabled on tablet.
			if ( ! empty( $row['style']['disable_tablet'] ) && $panels_tablet_width > $panels_mobile_width ) {
				$css->add_row_css( $post_id, $ri, null, array(
					'display' => 'none',
				), "$panels_tablet_width:$tablet_min_width" );
			}

			// Check if row is disabled on mobile.
			if ( ! empty( $row['style']['disable_mobile'] ) ) {
				$css->add_row_css( $post_id, $ri, null, array(
					'display' => 'none',
				), $panels_mobile_width );
			}

			foreach ( $row['cells'] as $ci => $cell ) {
				foreach ( $cell['widgets'] as $wi => $widget ) {
					// Check if widget is disabled on desktop.
					if ( ! empty( $widget['panels_info']['style']['disable_desktop'] ) ) {
						$css->add_widget_css( $post_id, $ri, $ci, $wi, null, array(
							'display' => 'none',
						), ":$desktop_breakpoint" );
					}

					// Check if widget is disabled on tablet.
					if ( ! empty( $widget['panels_info']['style']['disable_tablet'] ) && $panels_tablet_width > $panels_mobile_width ) {
						$css->add_widget_css( $post_id, $ri, $ci, $wi, null, array(
							'display' => 'none',
						), "$panels_tablet_width:$tablet_min_width" );
					}

					// Check if widget is disabled on mobile.
					if ( ! empty( $widget['panels_info']['style']['disable_mobile'] ) ) {
						$css->add_widget_css( $post_id, $ri, $ci, $wi, null, array(
							'display' => 'none',
						), $panels_mobile_width );
					}
				}
			}
		}

		return $css;
	}

	/**
	 * Conditionally filter a row from the layout based on visibility settings.
	 *
	 * Used if Page Builder version is newer than 2.16.7.
	 */
	public function maybe_hide_row( $output, $data ) {
		if ( is_admin() ) {
			return $output;
		}

		if ( empty( $data['style'] ) ) {
			return $output;
		}

		// Set up user role settings if they exist.
		if (
			! empty( $data['style']['toggle_user_role'] ) &&
			! empty( $data['style']['toggle_user_role_roles'] )
		) {
			$user_role_settings = array(
				'toggle_display' => $data['style']['toggle_user_role_toggle_display'] ?? 'show',
				'roles' => $data['style']['toggle_user_role_roles'],
			);
		}

		if (
			! empty( $data['style']['disable_row'] ) ||
			$this->addon->check_scheduling( $data['style'] ) ||
			(
				! empty( $data['style']['disable_logged_out'] ) &&
				! is_user_logged_in()
			) ||
			(
				! empty( $data['style']['disable_logged_in'] ) &&
				is_user_logged_in()
			) ||
			(
				! empty( $user_role_settings ) &&
				$this->addon->check_user_roles( $user_role_settings )
			)
		) {
			// Prevent row output.
			$output = false;
		}


		return $output;
	}


	/**
	 * Conditionally filter a widget from the layout based on visibility settings.
	 *
	 * Used if Page Builder version is newer than 2.16.7.
	 */
	public function maybe_hide_widget( $output, $data ) {
		if ( is_admin() ) {
			return $output;
		}

		if (
			empty( $data['panels_info'] ) ||
			empty( $data['panels_info']['style'] )
		) {
			return $output;
		}

		// Set up user role settings if they exist.
		if (
			! empty( $data['panels_info']['style']['toggle_user_role'] ) &&
			! empty( $data['panels_info']['style']['toggle_user_role_roles'] )
		) {
			$user_role_settings = array(
				'toggle_display' => $data['panels_info']['style']['toggle_user_role_toggle_display'] ?? 'show',
				'roles' => $data['panels_info']['style']['toggle_user_role_roles'],
			);
		}

		if (
			! empty( $data['panels_info']['style']['disable_widget'] ) ||
			$this->addon->check_scheduling( $data['panels_info']['style'] ) ||
			(
				! empty( $data['panels_info']['style']['disable_logged_out'] ) &&
				! is_user_logged_in()
			) ||
			(
				! empty( $data['panels_info']['style']['disable_logged_in'] ) &&
				is_user_logged_in()
			) ||
			(
				! empty( $user_role_settings ) &&
				$this->addon->check_user_roles( $user_role_settings )
			)
		) {
			// Prevent widget output.
			$output = false;
		}

		return $output;
	}
}