<?php
/*
Plugin Name: SiteOrigin Toggle Visibility
Description: Toggle visibility across Page Builder content or full pages, customizing access by device, logged in status, and time for strategic content arrangement.
Version: 1.0.0
Author: SiteOrigin
Author URI: https://siteorigin.com
License: GPL3
License URI: https://www.gnu.org/licenses/gpl-3.0.txt
Documentation: https://siteorigin.com/premium-documentation/plugin-addons/toggle-visibility
Tags: Page Builder
Requires: siteorigin-panels
*/

class SiteOrigin_Premium_Plugin_Toggle_Visibility {
	public $exempt_roles;
	public $pb_uses_toggle = false;
	const TEST_SHORTCODE_HAS_ROLES = true;

	public function __construct() {
		if (
			! defined( 'SITEORIGIN_PANELS_VERSION' ) ||
			! defined( 'SOW_BUNDLE_VERSION' )
		) {
			return;
		}

		$this->load_modules();

		// Assets.
		add_action( 'admin_print_scripts-appearance_page_so_panels_home_page', array( $this, 'enqueue_admin_assets' ), 20 );
		add_action( 'admin_print_scripts-post-new.php', array( $this, 'enqueue_admin_assets' ), 20 );
		add_action( 'admin_print_scripts-post.php', array( $this, 'enqueue_admin_assets' ), 20 );
		add_action( 'admin_print_scripts-widgets.php', array( $this, 'enqueue_admin_assets' ), 20 );

		// Shortcode.
		add_shortcode( 'toggle_visibility', array( $this, 'shortcode' ) );
	}

	public static function single() {
		static $single;

		return empty( $single ) ? $single = new self() : $single;
	}

	private function load_modules() {
		if ( class_exists( 'SiteOrigin_Premium_Plugin_Toggle_Visibility_Migrations' ) ) {
			return;
		}

		require plugin_dir_path( __FILE__ ) . 'inc/migrations.php';
		SiteOrigin_Premium_Plugin_Toggle_Visibility_Migrations::single();

		$this->pb_uses_toggle = SiteOrigin_Premium_Plugin_Toggle_Visibility_Migrations::single()->pb_uses_toggle;

		if ( defined( 'SITEORIGIN_PANELS_VERSION' ) ) {
			require plugin_dir_path( __FILE__ ) . 'inc/page-builder.php';
			SiteOrigin_Premium_Plugin_Toggle_Visibility_Page_Builder::single();
		}

		if ( defined( 'SOW_BUNDLE_VERSION' ) ) {
			require plugin_dir_path( __FILE__ ) . 'inc/metabox.php';
			SiteOrigin_Premium_Plugin_Toggle_Visibility_Metabox::single();
		}
	}

	public function enqueue_admin_assets() {
		if ( ! wp_script_is( 'sowb-pikaday-jquery' ) ) {
			// WB isn't active, load fallback scripts.
			wp_register_script(
				'sowb-pikaday',
				SITEORIGIN_PREMIUM_URL . 'js/pikaday' . SITEORIGIN_PREMIUM_JS_SUFFIX . '.js',
				array(),
				'1.6.1'
			);
			wp_register_script(
				'sowb-pikaday-jquery',
				SITEORIGIN_PREMIUM_URL . 'js/pikaday.jquery' . SITEORIGIN_PREMIUM_JS_SUFFIX . '.js',
				array( 'sowb-pikaday' ),
				'1.6.1'
			);

			wp_register_style(
				'sowb-pikaday-fallback',
				SITEORIGIN_PREMIUM_URL . 'css/pikaday-fallback.css',
				array()
			);

			wp_register_style(
				'sowb-pikaday',
				SITEORIGIN_PREMIUM_URL . 'css/pikaday.css',
				array( 'sowb-pikaday-fallback' ),
				'1.6.1'
			);
		}
		wp_enqueue_script( 'sowb-pikaday' );
		wp_enqueue_script( 'sowb-pikaday-jquery' );
		wp_enqueue_style( 'sowb-pikaday' );

		wp_enqueue_script(
			'siteorigin-premium-toggle-visibility-addon',
			plugin_dir_url( __FILE__ ) . 'js/script' . SITEORIGIN_PREMIUM_JS_SUFFIX . '.js',
			array( 'jquery', 'sowb-pikaday', 'sowb-pikaday-jquery' ),
			SITEORIGIN_PREMIUM_VERSION
		);

		wp_localize_script(
			'siteorigin-premium-toggle-visibility-addon',
			'soPremiumToggleVisibilityAddon',
			array(
				'isRTL' => is_rtl(),
				'i18n' => include SITEORIGIN_PREMIUM_DIR . 'inc/datapickeri18n.php',
				'useToggle' => $this->pb_uses_toggle,
			)
		);
	}

	/**
	 * Check if page/row/widget is scheduled to show or hide.
	 */
	public function check_scheduling( $styles ) {
		if ( ! $this->pb_uses_toggle ) {
			$styles = SiteOrigin_Premium_Plugin_Toggle_Visibility_Migrations::single()->pb_migrate_settings( $styles );
		}

		$toggle_display_name = 'toggle_scheduling_toggle_display';
		$date_from_name = 'toggle_scheduling_toggle_date_from';
		$date_to_name = 'toggle_scheduling_toggle_date_to';

		if ( empty( $styles[ $date_from_name ] ) && empty( $styles[ $date_to_name ] ) ) {
			return false;
		}

		$scheduled = false;
		$current_time = new DateTime( 'now', new DateTimeZone( wp_timezone_string() ) );
		$from = ! empty( $styles[ $date_from_name ] ) ? new DateTime( $styles[ $date_from_name ], new DateTimeZone( wp_timezone_string() ) ) : '';
		$to = ! empty( $styles[ $date_to_name ] ) ? new DateTime( $styles[ $date_to_name ] . ' 23:59', new DateTimeZone( wp_timezone_string() ) ) : '';

		if ( ! empty( $from ) && ! empty( $to ) ) {
			if ( empty( $from ) && $to < $current_time ) {
				$scheduled = true;
			} elseif ( $current_time > $from && $current_time < $to ) {
				$scheduled = true;
			}
		} elseif ( ! empty( $from ) && $current_time > $from ) {
			$scheduled = true;
		} elseif ( ! empty( $to ) && $current_time < $to ) {
			$scheduled = true;
		}
		$toggle_display_value = ! empty( $styles[ $toggle_display_name ] ) ? $styles[ $toggle_display_name ] : 'show';

		return $toggle_display_value == 'show' ? ! $scheduled : $scheduled;
	}

	public function shortcode( $atts, $content = '' ) {
		$atts = shortcode_atts(
			array(
				// Logged defaults to `in` to avoid a situation where something
				// is unintentionally publicly visible.
				'logged' => 'in',
				'roles' => '',
				'display' => 'hide',
			),
			$atts,
			'toggle_visibility'
		);

		if ( ! empty( $atts['roles'] ) && is_string( $atts['roles'] ) ) {
			$atts['roles'] = explode( ',', $atts['roles'] );
			$atts['roles'] = array_map( 'trim', $atts['roles'] );

			// Ensure the first role isn't empty.
			if ( empty( $atts['roles'] ) || empty( $atts['roles'][0] ) ) {
				return $content;
			}

			$block_content = $this->check_user_roles(
				array(
					'toggle_display' => $atts['display'],
					'roles' => $atts['roles'],
				)
			);

			return $block_content ? '' : $content;
		}

		if ( $atts['logged'] === 'in' && ! is_user_logged_in() ) {
			return '';
		}

		if ( $atts['logged'] === 'out' && is_user_logged_in() ) {
			return '';
		}

		return $content;
	}

	/**
	 * Retrieves a filtered list of user roles.
	 *
	 * This method fetches all available user roles using `SiteOrigin_Premium_Utility::get_roles()`, and excludes specified
	 * roles that are exempt from visibility checks.
	 *
	 * The roles to be removed can be filtered using the 'siteorigin_premium_toggle_visibility_exempt_roles' filter.
	 *
	 * @return array An associative array of roles, with role keys as
	 * array keys and role names as array values.
	 */
	public function get_roles() {
		$this->prepare_exempt_roles();
		return SiteOrigin_Premium_Utility::get_roles(
			$this->exempt_roles
		);
	}

	/**
	 * Prepare the list of exempt user roles.
	 *
	 * This function applies the 'siteorigin_premium_toggle_visibility_exempt_roles' filter
	 * to get the list of roles that are exempt from visibility checks.
	 * By default, the 'administrator' role is exempt. The result is
	 * cached in the $this->exempt_roles property.
	 *
	 * @return void
	 */
	public function prepare_exempt_roles() {
		if ( ! empty( $this->exempt_roles ) ) {
			return $this->exempt_roles;
		}

		$exempt_roles = apply_filters(
			'siteorigin_premium_toggle_visibility_exempt_roles',
			array(
				'administrator',
			)
		);

		if (
			empty( $exempt_roles ) ||
			! is_array( $exempt_roles )
		) {
			$this->exempt_roles = array();
			return;
		}

		$this->exempt_roles = array_map( 'esc_attr', $exempt_roles );
	}

	/**
	 * Determines if content should be blocked based on user roles.
	 *
	 * Exempt users are always allowed access regardless of the settings.
	 * The method returns a valid state based on the `$toggle_display` value.
	 *
	 * @param array $settings Contains 'toggle_display' and 'roles' for visibility control.
	 *
	 * @return bool True to block content, false to allow.
	 */
	public function check_user_roles( $settings ) {
		if ( empty( $settings ) ) {
			return true;
		}

		// If no roles are set, block access.
		if ( empty( $settings['roles'] ) || ! is_array( $settings['roles'] ) ) {
			return true;
		}

		// If the user isn't logged in, we can't check their role. Prevent access.
		if ( ! is_user_logged_in() ) {
			return true;
		}

		$user_role = wp_get_current_user()->roles;
		if ( empty( $user_role ) ) {
			return true;
		}

		$this->prepare_exempt_roles();
		// Exempt users are always allowed access.
		if (
			! empty( $this->exempt_roles ) &&
			array_intersect( $this->exempt_roles, $user_role )
		) {
			return false;
		}

		$block_content_if_role_found = $settings['toggle_display'] === 'hide';
		// Check if the current user has any of the flagged roles for visibility.
		foreach ( $settings['roles'] as $role ) {
			if ( in_array( $role, $user_role ) ) {
				return $block_content_if_role_found;
			}
		}

		return ! $block_content_if_role_found;
	}
}
