<?php

/**
 * This class handles rendering the SOWCTB (SiteOrigin WooCommerce Template
 * Builder) templates in the Full Site Editor.
 */
class SOWC_Full_Site_Editor_Override {
	private $rendered = false;
	private $so_wc_templates = array();
	private $wc_pages = array();
	private $template_type = '';

	public function __construct() {
		add_filter( 'render_block', array( $this, 'render_block' ), 10, 2 );
	}

	public static function single() {
		static $single;

		return empty( $single ) ? $single = new self() : $single;
	}

	/**
	 * Sets up and checks if WCTB settings exist.
	 * Loads templates from options table if not already loaded.
	 * Will also store the WC template page ids.
	 *
	 * @return boolean True if valid templates exist, false otherwise.
	 */
	public function setup_and_check_data(): bool {
		if ( ! empty( $this->so_wc_templates ) ) {
			return true;
		}

		$this->so_wc_templates = get_option( 'so-wc-templates' );

		$this->wc_pages = array(
			wc_get_page_id( 'cart' ) => 'cart',
			wc_get_page_id( 'checkout' ) => 'checkout',
			wc_get_page_id( 'myaccount' ) => 'myaccount',
		);

		return ! empty( $this->so_wc_templates ) && is_array( $this->so_wc_templates );
	}

	/**
	 * Checks if a WCTB template should be applied to the current product.
	 * Sets the template ID as a query var if found to help with rendering.
	 *
	 * @return boolean|null False if no template applies, null if a template is set.
	 */
	public function check_if_wctb_present() {
		if ( ! $this->setup_and_check_data() ) {
			return false;
		}

		if ( is_embed() ) {
			return false;
		}

		$current_page_id = get_the_ID();

		if ( is_product() ) {
			if ( $this->set_product_template_id() ) {
				$this->template_type = 'content-single-product';
			}
		} elseif ( is_shop() ) {
			if ( ! empty( $this->so_wc_templates['shop']['active'] ) ) {
				$this->template_type = 'shop';
			}
		} elseif ( is_product_category() || is_product_tag() ) {
			$this->set_archive_template_type( $current_page_id );
		} elseif ( isset( $this->wc_pages[ $current_page_id ] ) ) {
			$this->set_non_product_archive_template_type(
				$this->wc_pages[ $current_page_id ]
			);
		}

		return ! empty( $this->template_type );
	}

	/**
	 * Retrieve the template ID for the current product.
	 *
	 * Determines whether a product-specific template is assigned via metadata.
	 * If no specific template is found, it falls back to a global template ID.
	 * Sets the template ID as a query variable for rendering purposes.
	 *
	 * @return bool True if a template ID is found and set, false otherwise.
	 */
	private function set_product_template_id(): bool {
		// Detect whether a global template, or a product specific template is being used.
		$template_id = get_post_meta( get_the_ID(), 'so_wc_template_post_id', true );
		if (
			empty( $template_id ) &&
			! empty( $this->so_wc_templates['content-single-product']['post_id'] )
		) {
			$template_id = $this->so_wc_templates['content-single-product']['post_id'];
		}

		if ( empty( $template_id ) ) {
			return false;
		}

		set_query_var( 'wctb_template_id', $template_id );
		return true;
	}

	/**
	 * Set the template type for product archive pages.
	 *
	 * Determines the appropriate template type for product category or tag archives.
	 * Uses taxonomy metadata to find the active template and sets the template type
	 * for rendering purposes.
	 *
	 * @return void
	 */
	private function set_archive_template_type(): void {
		// Set up the $term if it is not already set.
		if ( empty( get_queried_object() ) ) {
			$term = get_term_by(
				'slug',
				get_query_var( 'term' ),
				get_query_var( 'taxonomy' )
			);
		} else {
			$term = get_queried_object();
		}

		$template = SiteOrigin_Premium_Plugin_WooCommerce_Templates::single()->set_active_template(
			! empty( $term ) && is_object( $term )
				? get_option( "_term_type_{ $term->taxonomy }_{ $term->term_id }" )
				: '',
			$this->so_wc_templates['content-product']
		);

		if ( empty( $template ) ) {
			return;
		}

		$this->template_type = 'archive';
	}

	/**
	 * Set the template type for non-product pages, or archives.
	 *
	 * Determines the appropriate template type based on the current page context,
	 * such as checkout or cart. Handles fallback scenarios for inactive templates.
	 *
	 * @param string $type The initial template type.
	 *
	 * @return bool True if the template type is set, false otherwise.
	 */
	private function set_non_product_archive_template_type( $type ): bool {
		// Adjust template type for specific scenarios.
		if (
			$type === 'checkout' &&
			is_wc_endpoint_url( 'order-received' )
		) {
			$type = 'thankyou';
		} elseif (
			$type === 'cart' &&
			! empty( WC()->cart ) &&
			WC()->cart->is_empty() &&
			! empty( $this->so_wc_templates['cart-empty']['active'] )
		) {
			$type = 'cart-empty';
		}

		// Check if the template type is active.
		if ( empty( $this->so_wc_templates[ $type ]['active'] ) ) {
			return false;
		}

		$this->template_type = $type;
		return true;
	}

	/**
	 * Set up the cart for the thank-you page template.
	 *
	 * This method prepares the thank-you page by clearing the
	 * cart and repopulating it with the items from the order.
	 * It then initiates the checkout process for rendering.
	 *
	 * @return WC_Checkout|false The checkout object if successful, or
	 * false if the order is invalid.
	 */
	private function setup_thank_you() {
		$order_id = get_query_var( 'order-pay' );
		$order = wc_get_order( $order_id );

		if ( empty( $order ) ) {
			return false;
		}

		WC()->cart->empty_cart();

		foreach ( $order->get_items() as $item ) {
			$product = $item->get_product();

			if ( empty( $product ) ) {
				continue;
			}

			WC()->cart->add_to_cart(
				$product->get_id(),
				$item->get_quantity()
			);
		}

		return WC()->checkout();
	}

	/**
	 * Renders the WCTB template and inserts it into the block content.
	 * Preserves the <main> tag while replacing its contents.
	 *
	 * @param string $content The original block content.
	 *
	 * @return string Modified content with the template injected.
	 */
	public function render_wctb_template( $content ): string {
		$this->rendered = true;

		add_filter( 'so_wc_wrapper_classes', array( $this, 'add_block_editor_wrapper_classes' ), 10, 1 );

		if ( $this->template_type === 'thankyou' ) {
			$checkout = $this->setup_thank_you();
		} elseif ( $this->template_type === 'myaccount' ) {
			$this->template_type = 'my-account';
		}

		// Set a flag to prevent certain actions from running in the block editor.
		// This is necessary to avoid WordPress theme error flags.
		$block_editor_output = true;

		ob_start();
		require SiteOrigin_Premium::dir_path( __FILE__ ) . '../templates/' . sanitize_file_name( $this->template_type ) . '.php';
		$rendered_template = ob_get_clean();

		wp_enqueue_style( 'so-wc-content-product-single' );

		// Replace the contents of the <main> tag with the rendered
		// WCTB template while preserving the tag structure.
		return preg_replace(
			'/(<main[^>]*>).*?(<\/main>)/s',
			'$1' . $rendered_template . '$2',
			$content
		);
	}

	/**
	 * Adds 'alignwide' class to the WCTB wrapper classes.
	 *
	 * @param array $classes Existing wrapper classes.
	 *
	 * @return array Modified wrapper classes with 'alignwide' added.
	 */
	public function add_block_editor_wrapper_classes( $classes ): array {
		$classes[] = 'alignwide';
		return $classes;
	}

	/**
	 * Determines if a WCTB template should be applied based on the
	 * current page context.
	 *
	 * @param string $content Block content.
	 * @param array $attributes Block attributes.
	 *
	 * @return string Modified or original block content.
	 */
	public function render_block( $content, $attributes ): string {
		if ( $this->rendered ) {
			return $content;
		}

		if (
			empty( $attributes ) ||
			empty( $attributes['attrs'] ) ||
			empty( $attributes['attrs']['tagName'] ) ||
			$attributes['attrs']['tagName'] !== 'main'
		) {
			return $content;
		}

		if ( $this->check_if_wctb_present() === false ) {
			return $content;
		}

		return $this->render_wctb_template( $content );
	}
}
