<?php
class SiteOrigin_Premium_Plugin_Toggle_Visibility_Metabox {
	public $premiumMeta = array();
	private $addon;
	private $migrations;

	public function __construct() {
		$this->addon = SiteOrigin_Premium_Plugin_Toggle_Visibility::single();
		$this->migrations = SiteOrigin_Premium_Plugin_Toggle_Visibility_Migrations::single();

		add_filter( 'siteorigin_premium_metabox_form_options', array( $this, 'add_metabox' ), 1, 1 );
		add_filter( 'the_content', array( $this, 'content_visibility' ), 99 );
		add_filter( 'template_redirect', array( $this, 'page_visibility' ), 1 );
	}

	public static function single() {
		static $single;

		return empty( $single ) ? $single = new self() : $single;
	}

	public function add_metabox( $form_options ) {
		$form_options += array(
			'toggle_visibility' => array(
				'type' => 'section',
				'label' => __( 'Page Visibility', 'siteorigin-premium' ),
				'tab' => true,
				'hide' => true,
				'fields' => array(
					'target' => array(
						'type' => 'radio',
						'label' => __( 'Visibility Target', 'siteorigin-premium' ),
						'default' => 'none',
						'options' => array(
							'none' => __( 'None', 'siteorigin-premium' ),
							'content' => __( 'Content', 'siteorigin-premium' ),
							'page' => __( 'Page', 'siteorigin-premium' ),
						),
						'state_emitter' => array(
							'callback' => 'select',
							'args' => array( 'visibility_target' ),
						),
					),

					'action' => array(
						'type' => 'radio',
						'label' => __( 'Action', 'siteorigin-premium' ),
						'default' => 'show',
						'options' => array(
							'show' => __( 'Show', 'siteorigin-premium' ),
							'disabled' => __( 'Hide', 'siteorigin-premium' ),
							'disable_logged_in' => __( 'Hide Logged In', 'siteorigin-premium' ),
							'disable_logged_out' => __( 'Hide Logged Out', 'siteorigin-premium' ),
						),
						'state_handler' => array(
							'visibility_target[none]' => array( 'hide' ),
							'_else[visibility_target]' => array( 'show' ),
						),
						'state_emitter' => array(
							'callback' => 'select',
							'args' => array( 'visibility' ),
						),
					),

					'redirect' => array(
						'type' => 'link',
						'label' => __( 'Redirect URL', 'siteorigin-premium' ),
						'description' => __( 'Optionally redirect the user to a different URL when the page is hidden.', 'siteorigin-premium' ),
						'state_handler' => array(
							'visibility_target[page]' => array( 'show' ),
							'_else[visibility_target]' => array( 'hide' ),
						),
					),

					'content' => array(
						'type' => 'section',
						'label' => __( 'Content', 'siteorigin-premium' ),
						'hide' => true,
						'state_handler' => array(
							'visibility_target[content]' => array( 'show' ),
							'_else[visibility_target]' => array( 'hide' ),
						),
						'fields' => array(
							'content_type' => array(
								'type' => 'radio',
								'label' => __( 'Content Type', 'siteorigin-premium' ),
								'default' => 'text',
								'options' => array(
									'text' => __( 'Text', 'siteorigin-premium' ),
									'builder' => __( 'Layout Builder', 'siteorigin-premium' ),
								),
								'state_emitter' => array(
									'callback' => 'select',
									'args' => array( 'content_type' ),
								),
							),
							'message_tinymce' => array(
								'type' => 'tinymce',
								'label' => __( 'Content', 'siteorigin-premium' ),
								'description' => __( 'The message to display when the content is hidden.', 'siteorigin-premium' ),
								'state_handler' => array(
									'content_type[text]' => array( 'show' ),
									'_else[content_type]' => array( 'hide' ),
								),
								'default' => '<em>' . __( 'Page content currently unavailable for viewing.', 'siteorigin-premium' ) . '</em>',
							),
							'message_builder' => array(
								'type' => 'builder',
								'label' => __( 'Content', 'siteorigin-premium' ),
								'description' => __( 'The message to display when the content is hidden.', 'siteorigin-premium' ),
								'state_handler' => array(
									'content_type[builder]' => array( 'show' ),
									'_else[content_type]' => array( 'hide' ),
								),
							),
						),
					),

					'advanced' => array(
						'type' => 'section',
						'label' => __( 'Advanced', 'siteorigin-premium' ),
						'hide' => true,
						'state_handler' => array(
							'visibility_target[none]' => array( 'hide' ),
							'_else[visibility_target]' => array( 'show' ),
						),
						'fields' => array(
							'scheduling' => array(
								'type' => 'section',
								'label' => __( 'Scheduling', 'siteorigin-premium' ),
								'hide' => true,
								'fields' => array(
									'toggle_date_from' => array(
										'label' => __( 'Date From', 'siteorigin-premium' ),
										'type' => 'text',
									),
									'toggle_date_to' => array(
										'label' => __( 'Date To', 'siteorigin-premium' ),
										'type' => 'text',
									),
								),
							),
						),
					),
				),
			),
		);

		// To resolve a saving issue with the user roles field,
		// the user will need a newer version of WB.
		if ( ! $this->migrations->wb_user_roles_support ) {
			return $form_options;
		}

		siteorigin_widgets_array_insert(
			$form_options['toggle_visibility']['fields']['advanced']['fields'],
			'scheduling',
			array(
				'user_roles' => array(
					'type' => 'select',
					'multiple' => true,
					'select2' => true,
					'label' => __( 'User Roles', 'siteorigin-premium' ),
					'description' => __( 'Apply Toggle Visibility to specific roles.', 'siteorigin-premium' ),
					'state_handler' => array(
						'visibility[disable_logged_in,disable_logged_out]' => array( 'hide' ),
						'_else[visibility]' => array( 'show' ),
					),
					'options' => $this->addon->get_roles(),
				),
			)
		);

		// Add a description to the user roles field if there are exempt roles.
		if ( ! empty( $this->addon->exempt_roles ) ) {
			$form_options['toggle_visibility']['fields']['advanced']['fields']['user_roles']['description'] = SiteOrigin_Premium_Utility::get_user_roles_description( $this->addon->exempt_roles );
		}

		return $form_options;
	}

	public function load_meta() {
		$this->premiumMeta = get_post_meta( get_the_ID(), 'siteorigin_premium_meta', true );

		if (
			empty( $this->premiumMeta ) ||
			empty( $this->premiumMeta['toggle_visibility'] )
		) {
			return;
		}

		if ( $this->migrations->wb_old_metabox ) {
			$migrated_instance = apply_filters(
				'siteorigin_premium_metabox_modify_instance',
				$this->premiumMeta
			);
			$this->premiumMeta = $migrated_instance['toggle_visibility'];
		} else {
			$this->premiumMeta = $this->premiumMeta['toggle_visibility'];
		}
	}

	/**
	 * Check if the current time is within the scheduled period.
	 *
	 * This method returns true if the content can't be scheduled, or if it is scheduled
	 * and the current time is within the scheduled period. It returns false if the content
	 * is scheduled but the current time is not within the scheduled period.
	 *
	 * @return bool True if the content can't be scheduled or is within the scheduled period, false otherwise.
	 */
	public function is_within_scheduled_period() {
		if (
			empty( $this->premiumMeta['advanced'] ) ||
			empty( $this->premiumMeta['advanced']['scheduling'] ) ||
			! is_array( $this->premiumMeta['advanced']['scheduling'] ) ||
			empty( $this->premiumMeta['advanced']['scheduling']['toggle_date_from'] ) ||
			empty( $this->premiumMeta['advanced']['scheduling']['toggle_date_to'] )
		) {
			return $this->premiumMeta['action'] !== 'show';
		}

		return $this->addon->check_scheduling(
			array(
				'toggle_display' => $this->premiumMeta['action'],
				'toggle_date_from' => $this->premiumMeta['advanced']['scheduling']['toggle_date_from'],
				'toggle_date_to' => $this->premiumMeta['advanced']['scheduling']['toggle_date_to'],
			)
		);
	}

	/**
	 * Check if the current user's role is blocked.
	 *
	 * This method returns true if the user's role is a blocked role.
	 * It returns false if the user's role is not blocked, or if there isn't
	 * a user role set.
	 *
	 * @return bool True if the user's role is blocked, false otherwise.
	 */
	public function is_user_role_blocked() {
		if (
			empty( $this->premiumMeta['advanced'] ) ||
			empty( $this->premiumMeta['advanced']['user_roles'] )
		) {
			return $this->premiumMeta['action'] === 'show';
		}

		return $this->addon->check_user_roles(
			$this->premiumMeta['advanced']['user_roles']
		);
	}

	public function maybe_hide_context( $context ) {
		if (
			empty( $this->premiumMeta ) ||
			empty( $this->premiumMeta['target'] ) ||
			$this->premiumMeta['target'] !== $context
		) {
			return;
		}

		$schedule_status = $this->is_within_scheduled_period();

		if ( $this->premiumMeta['action'] === 'disable_logged_out' ) {
			return $schedule_status && ! is_user_logged_in();
		}

		if ( $this->premiumMeta['action'] === 'disable_logged_in' ) {
			return $schedule_status && is_user_logged_in();
		}

		if ( ! $this->is_user_role_blocked() ) {
			return false;
		}

		// Page is set to hidden.
		if ( $this->premiumMeta['action'] === 'disabled' ) {
			return $schedule_status;
		}

		return $schedule_status;
	}

	public function content_visibility( $content ) {
		$this->load_meta();

		if ( ! $this->maybe_hide_context( 'content' ) ) {
			return $content;
		}
		add_filter( 'siteorigin_panels_filter_content_enabled', '__return_false' );
		$meta_content = $this->premiumMeta['content'];

		// Detect the selected content type, and check if there's valid content.
		if (
			$meta_content['content_type'] == 'text' &&
			! empty( $meta_content['message_tinymce'] )
		) {
			$content = $meta_content['message_tinymce'];
		} elseif (
			$meta_content['content_type'] == 'builder' &&
			! empty( $meta_content['message_builder'] )
		) {
			$content = siteorigin_panels_render(
				'w' . get_the_ID(),
				true,
				$meta_content['message_builder']
			);
		} else {
			// If there isn't valid content, fallback to a default message.
			$content = apply_filters(
				'siteorigin_premium_toggle_visibility_metabox_content_fallback',
				esc_html__( 'This content has been hidden', 'siteorigin-premium' ),
				$this->premiumMeta
			);
		}

		// Now that we've generated the block message, allow developers to
		// modify it.
		return apply_filters(
			'siteorigin_premium_toggle_visibility_metabox_block_message',
			$content,
			$this->premiumMeta
		);
	}

	public function page_visibility() {
		$this->load_meta();

		if ( ! $this->maybe_hide_context( 'page' ) ) {
			return;
		}

		if ( ! empty( $this->premiumMeta['redirect'] ) ) {
			wp_redirect( sow_esc_url(
				do_shortcode(
					$this->premiumMeta['redirect']
				)
			) );

			die();
		}

		global $wp_query;
		$wp_query->set_404();
		status_header( 404 );
		nocache_headers();
	}
}
