<?php
class SiteOrigin_Premium_Plugin_Toggle_Visibility_Migrations {
	public $pb_uses_new_widget_row_hiding = true;
	public $pb_uses_toggle = false;
	public $pb_user_roles_support = false;
	public $wb_user_roles_support = false;
	public $wb_old_metabox = false;

	public function __construct() {
		if ( defined( 'SOW_BUNDLE_VERSION' ) ) {
			$this->metabox_migrations();
		}

		if ( defined( 'SITEORIGIN_PANELS_VERSION' ) ) {
			$this->pb_migrations();
		}
	}

	public static function single() {
		static $single;

		return empty( $single ) ? $single = new self() : $single;
	}

	public function metabox_migrations() {
		// There's an issue with older versions of WB that can cause the
		// multi-select field to not work correctly.
		$this->wb_user_roles_support = version_compare( SOW_BUNDLE_VERSION, '1.64.0', '>=' );

		$this->wb_old_metabox = true;
		add_filter( 'siteorigin_premium_metabox_modify_instance', array( $this, 'metabox_migrate_to_new_structure' ), 10, 2 );
	}

	public function pb_migrations() {
		// Older versions of PB don't include a multi-select filed.
		$this->pb_user_roles_support = version_compare( SITEORIGIN_PANELS_VERSION, '2.31.0', '>=' );

		if ( version_compare( SITEORIGIN_PANELS_VERSION, '2.16.6', '<=' ) ) {
			$this->pb_uses_new_widget_row_hiding = false;
			add_filter( 'siteorigin_panels_layout_data', array( $this, 'pb_filter_row_widget' ), 10, 2 );
		}

		// If a newer version of PB is active, we need to migrate the
		// schedule related settings.
		if ( version_compare( SITEORIGIN_PANELS_VERSION, '2.17.0', '>=' ) ) {
			$this->pb_uses_toggle = true;

			add_filter( 'siteorigin_panels_general_current_styles', array( $this, 'pb_migrate_settings' ), 10, 1 );
			add_filter( 'siteorigin_panels_general_style_fields', array( $this, 'pb_migrate_settings_pre_save' ), 10, 1 );
			add_filter( 'siteorigin_panels_data_pre_save', array( $this, 'pb_migrate_settings_save' ), 10, 1 );
		}
	}

	/**
	 * Migrate the metabox settings to the new structure.
	 *
	 * This method migrates the metabox to align with the updated structure.
	 * It performs the following operations:
	 * - Renames the `status` field to `action`.
	 * - Moves scheduling and user roles data into the `advanced` array.
	 * - Removes deprecated fields that are no longer needed.
	 * - Ensures locked actions (`disabled`, `disable_logged_in`, `disable_logged_out`) remain unaffected by user roles or scheduling settings.
	 *
	 * @param array $instance The instance settings to be migrated. This array contains the `toggle_visibility` settings.
	 *
	 * @return array The migrated instance settings.
	 */
	public function metabox_migrate_to_new_structure( $instance ) {
		if (
			! isset( $instance['toggle_visibility'] ) ||
			! is_array( $instance['toggle_visibility'] )
		) {
			return $instance;
		}

		$settings = $instance['toggle_visibility'];

		// Check if this data has already been migrated - if it has action but no status, it's already migrated.
		if (
			isset( $settings['action'] ) &&
			! isset( $settings['status'] )
		) {
			return $instance;
		}

		$settings['advanced'] = array(
			'user_roles' => array(),
			'scheduling' => array(),
		);

		// If the status isn't set, default to disabled.
		$settings['action'] = empty( $settings['status'] )
		? 'disabled'
		: $settings['status'];
		unset( $settings['status'] );

		$initial_action = $settings['action'];

		// User Roles.
		if (
			isset( $settings['toggle_user_roles_data'] ) &&
			is_array( $settings['toggle_user_roles_data'] ) &&
			! empty( $settings['toggle_user_roles_data']['roles'] )
		) {
			if ( $initial_action === 'user_roles' ) {
				$settings['action'] = $settings['toggle_user_roles_data']['toggle_display'];
			}
			unset( $settings['toggle_user_roles_data']['toggle_display'] );

			// Copy everything else over.
			$settings['advanced']['user_roles'] = $settings['toggle_user_roles_data']['roles'];

			$user_role_migrated = true;

			unset( $settings['toggle_user_roles_data'] );
		}

		// Schedule.
		if (
			isset( $settings['toggle_scheduling_data'] ) &&
			is_array( $settings['toggle_scheduling_data'] ) &&
			(
				! empty( $settings['toggle_scheduling_data']['toggle_date_from'] ) ||
				! empty( $settings['toggle_scheduling_data']['toggle_date_to'] )
			)
		) {
			if ( $initial_action === 'scheduled' ) {
				$settings['action'] = $settings['toggle_scheduling_data']['toggle_display'];
			}
			unset( $settings['toggle_scheduling_data']['toggle_display'] );

			// Copy everything else over.
			$settings['advanced']['scheduling'] = $settings['toggle_scheduling_data'];
			$scheduled_migrated = true;

			unset( $settings['toggle_scheduling_data'] );
		}

		// The `hide` action is now `disabled`.
		if ( $settings['action'] === 'hide' ) {
			$settings['action'] = 'disabled';
		}

		// Should the Page Target be set to `None`?
		if (
			$settings['target'] === 'content' &&
			$settings['action'] === 'show' &&
			empty( $scheduled_migrated ) &&
			empty( $user_role_migrated )
		) {
			$settings['target'] = 'none';
		}

		$instance['toggle_visibility'] = $settings;

		return $instance;
	}

	/**
	 * Migrate Page Builder settings to the new structure.
	 *
	 * This method transforms legacy scheduling settings into the new structure.
	 * It also removes the old fields if the data is being saved via a POST
	 * request.
	 *
	 * @param array $style The style settings to be migrated. Includes legacy fields such as:
	 *                     - 'toggle_display' (string): Visibility toggle ('show' or 'hide').
	 *                     - 'toggle_date_from' (string): Start date for visibility.
	 *                     - 'toggle_date_to' (string): End date for visibility.
	 *
	 * @return array The migrated style settings with new field names and
	 *               old fields removed (if applicable).
	 */
	public function pb_migrate_settings( $style ) {
		if (
			empty( $style['toggle_display'] ) &&
			empty( $style['toggle_date_from'] ) &&
			empty( $style['toggle_date_to'] )
		) {
			return $style;
		}

		$style['toggle_scheduling_toggle_display'] = ! empty( $style['toggle_display'] ) ? $style['toggle_display'] : '';
		$style['toggle_scheduling_toggle_date_from'] = ! empty( $style['toggle_date_from'] ) ? $style['toggle_date_from'] : '';
		$style['toggle_scheduling_toggle_date_to'] = ! empty( $style['toggle_date_to'] ) ? $style['toggle_date_to'] : '';

		// If the user is saving, remove the old fields.
		if ( ! empty( $_POST ) ) {
			unset( $style['toggle_display'] );
			unset( $style['toggle_date_from'] );
			unset( $style['toggle_date_to'] );
		}

		return $style;
	}

	/**
	 * Migrate PB settings before saving.
	 *
	 * This method temporarily adds legacy fields during the save process
	 * to ensure that the new settings are saved correctly. Without these
	 * legacy fields, Page Builder will disregard any "invalid" fields,
	 * and the new fields will be considered invalid if the user doesn't
	 * open the row or widget.
	 *
	 * @param array $fields The fields to be saved.
	 *
	 * @return array The modified fields with legacy fields added.
	*/
	public function pb_migrate_settings_pre_save( $fields ) {
		if (
			! empty( $_POST ) &&
			! empty( $_POST['action'] ) &&
			$_POST['action'] != 'so_panels_style_form' &&
			! empty( $fields['toggle_scheduling'] )
		) {
			$fields['toggle_display'] = $fields['toggle_scheduling']['fields']['toggle_display'];
			$fields['toggle_date_from'] = $fields['toggle_scheduling']['fields']['toggle_date_from'];
			$fields['toggle_date_to'] = $fields['toggle_scheduling']['fields']['toggle_date_to'];
		}

		return $fields;
	}

	/**
	 * Migrate PB settings during the save process.
	 *
	 * This method iterates over the grids and widgets in the panels data and applies
	 * the migration settings to ensure that the new settings are saved correctly.
	 *
	 * @param array $panels_data The panels data to be saved.
	 *
	 * @return array The modified panels data with migrated settings.
	 */
	public function pb_migrate_settings_save( $panels_data ) {
		if ( ! empty( $panels_data['grids'] ) ) {
			foreach ( $panels_data['grids'] as $k => $row ) {
				$panels_data['grids'][ $k ]['style'] = $this->pb_migrate_settings(
					$panels_data['grids'][ $k ]['style']
				);
			}
		}

		if ( ! empty( $panels_data['widgets'] ) ) {
			foreach ( $panels_data['widgets'] as $k => $row ) {
				$panels_data['widgets'][ $k ]['panels_info']['style'] = $this->pb_migrate_settings(
					$panels_data['widgets'][ $k ]['panels_info']['style']
				);
			}
		}

		return $panels_data;
	}

	/**
	 * Conditionally filter a row/widget from the layout based on visibility settings.
	 * LEGACY: Used if Page Builder version is older than 2.16.8.
	 */
	public function pb_filter_row_widget( $layout_data ) {
		$addon = SiteOrigin_Premium_Plugin_Toggle_Visibility::single();

		// Row Visibility.
		foreach ( $layout_data as $ri => $row ) {
			if (
				! empty( $row['style']['disable_row'] ) ||
				$addon->check_scheduling( $row['style'] ) ||
				(
					! empty( $row['style']['disable_logged_out'] ) &&
					! is_user_logged_in()
				) ||
				(
					! empty( $row['style']['disable_logged_in'] ) &&
					is_user_logged_in()
				)
			) {
				// Prevent row output.
				unset( $layout_data[ $ri ] );
			}

			foreach ( $row['cells'] as $ci => $cell ) {
				// Widget Visibility.
				foreach ( $cell['widgets'] as $wi => $widget ) {
					if (
						! isset( $widget['panels_info'] ) ||
						! isset( $widget['panels_info']['style'] )
					) {
						continue;
					}

					if (
						! empty( $widget['panels_info']['style']['disable_widget'] ) ||
						$addon->check_scheduling( $widget['panels_info']['style'] ) ||
						(
							! empty( $widget['panels_info']['style']['disable_logged_out'] ) &&
							! is_user_logged_in()
						) ||
						(
							! empty( $widget['panels_info']['style']['disable_logged_in'] ) &&
							is_user_logged_in()
						)
					) {
						// Prevent widget output.
						unset( $layout_data[ $ri ]['cells'][ $ci ]['widgets'][ $wi ] );
					}
				}
			}
		}

		return $layout_data;
	}
}
